/**
 * Site Builder Frontend JS
 * Handles form submission and progress display
 */

const form = document.getElementById('websiteForm');
const formCard = document.getElementById('formCard');
const successCard = document.getElementById('successCard');
const loadingOverlay = document.getElementById('loadingOverlay');
const loadingStatus = document.getElementById('loadingStatus');
const progressSteps = document.querySelectorAll('.progress-step');
const errorMessage = document.getElementById('errorMessage');
const downloadBtn = document.getElementById('downloadBtn');
const submitBtn = document.getElementById('submitBtn');

// Form submission
form.addEventListener('submit', async (e) => {
    e.preventDefault();
    hideError();

    const formData = {
        domain: document.getElementById('domain').value.trim().toLowerCase(),
        business_type: document.getElementById('businessType').value,
        industry: document.getElementById('industry').value.trim()
    };

    // Validate
    if (!validateForm(formData)) {
        return;
    }

    // Disable button
    submitBtn.disabled = true;

    // Show loading
    showLoading();

    try {
        // Start generation
        const response = await fetch('api/generate.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(formData)
        });

        // Poll for progress
        pollProgress(formData.domain);

        const result = await response.json();

        if (!response.ok) {
            throw new Error(result.error || 'Generation failed');
        }

        if (result.success && result.download_url) {
            // Complete all steps
            completeAllSteps();
            
            // Show success
            setTimeout(() => {
                hideLoading();
                showSuccess(result.download_url);
            }, 500);
        } else {
            throw new Error(result.error || 'Unknown error occurred');
        }

    } catch (error) {
        console.error('Generation error:', error);
        hideLoading();
        showError(error.message || 'Failed to generate website. Please try again.');
        submitBtn.disabled = false;
    }
});

// Validation
function validateForm(data) {
    // Domain validation - simple check for at least one dot
    if (!data.domain || !data.domain.includes('.') || data.domain.length < 4) {
        showError('Please enter a valid domain (e.g., mybusiness.com)');
        document.getElementById('domain').classList.add('error');
        return false;
    }

    // Business type
    if (!data.business_type) {
        showError('Please select a business type');
        document.getElementById('businessType').classList.add('error');
        return false;
    }

    // Industry
    if (data.industry.length < 5) {
        showError('Please enter a more specific industry/niche (at least 5 characters)');
        document.getElementById('industry').classList.add('error');
        return false;
    }

    return true;
}

// Progress polling (optional - for real-time updates)
async function pollProgress(domain) {
    const steps = [
        { step: 1, delay: 2000 },
        { step: 2, delay: 3000 },
        { step: 3, delay: 4000 },
        { step: 4, delay: 2000 },
        { step: 5, delay: 1500 }
    ];

    const statuses = [
        'Generating unique content...',
        'Applying design template...',
        'Downloading images from Unsplash...',
        'Building HTML, CSS files...',
        'Creating ZIP package...'
    ];

    for (let i = 0; i < steps.length; i++) {
        await sleep(steps[i].delay);
        updateStep(steps[i].step);
        loadingStatus.textContent = statuses[i];
    }
}

// Update progress step
function updateStep(currentStep) {
    progressSteps.forEach(el => {
        const stepNum = parseInt(el.dataset.step);
        if (stepNum < currentStep) {
            el.classList.remove('active');
            el.classList.add('done');
            el.querySelector('.step-icon').textContent = '✓';
        } else if (stepNum === currentStep) {
            el.classList.add('active');
            el.classList.remove('done');
        } else {
            el.classList.remove('active', 'done');
        }
    });
}

// Complete all steps
function completeAllSteps() {
    progressSteps.forEach(el => {
        el.classList.remove('active');
        el.classList.add('done');
        el.querySelector('.step-icon').textContent = '✓';
    });
    loadingStatus.textContent = 'Complete!';
}

// UI helpers
function showLoading() {
    loadingOverlay.classList.add('active');
    resetProgressSteps();
}

function hideLoading() {
    loadingOverlay.classList.remove('active');
}

function showSuccess(downloadUrl) {
    formCard.classList.add('hidden');
    successCard.classList.remove('hidden');
    downloadBtn.href = downloadUrl;
}

function showError(message) {
    errorMessage.textContent = message;
    errorMessage.classList.add('show');
}

function hideError() {
    errorMessage.classList.remove('show');
    document.querySelectorAll('.form-input, .form-select').forEach(el => {
        el.classList.remove('error');
    });
}

function resetProgressSteps() {
    progressSteps.forEach((el, index) => {
        el.classList.remove('active', 'done');
        el.querySelector('.step-icon').textContent = index + 1;
    });
}

function sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

// Reset form
function resetForm() {
    form.reset();
    formCard.classList.remove('hidden');
    successCard.classList.add('hidden');
    submitBtn.disabled = false;
    hideError();
    resetProgressSteps();
}

// Remove error class on input
document.querySelectorAll('.form-input, .form-select').forEach(el => {
    el.addEventListener('input', () => {
        el.classList.remove('error');
        hideError();
    });
});
