<?php
/**
 * Site Builder API - Generate Website
 * 
 * POST /api/generate.php
 * 
 * Input:
 * - domain: string
 * - business_type: string
 * - industry: string
 * 
 * Output:
 * - success: boolean
 * - download_url: string (on success)
 * - error: string (on failure)
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/ContentGenerator.php';
require_once __DIR__ . '/TemplateEngine.php';
require_once __DIR__ . '/ImageFetcher.php';
require_once __DIR__ . '/ZipBuilder.php';

// Only POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Get input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid JSON input']);
    exit;
}

// Validate input
$domain = sanitizeDomain($input['domain'] ?? '');
$businessType = sanitizeString($input['business_type'] ?? '');
$industry = sanitizeString($input['industry'] ?? '');

if (!$domain || !$businessType || !$industry) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required fields: domain, business_type, industry']);
    exit;
}

if (!isValidDomain($domain)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid domain format']);
    exit;
}

try {
    // Clean up old files
    cleanupOldFiles();
    
    // Create unique ID for this generation
    $generationId = md5($domain . time() . random_bytes(8));
    $outputDir = GENERATED_PATH . '/' . $generationId;
    
    if (!mkdir($outputDir, 0755, true)) {
        throw new Exception('Failed to create output directory');
    }
    
    // Step 1: Generate content with ChatGPT
    $contentGenerator = new ContentGenerator(OPENAI_API_KEY);
    $content = $contentGenerator->generate($domain, $businessType, $industry);
    
    if (!$content) {
        throw new Exception('Failed to generate content');
    }
    
    // Step 2: Select and apply template
    $templateEngine = new TemplateEngine(TEMPLATES_PATH);
    $template = $templateEngine->selectTemplate($businessType);
    
    // Step 3: Fetch images
    $imageFetcher = new ImageFetcher(UNSPLASH_ACCESS_KEY);
    $images = $imageFetcher->fetchImages($industry, $businessType, 5);
    
    // Step 4: Build website files
    $files = $templateEngine->build($content, $template, $images, $domain);
    
    // Write files to output directory
    foreach ($files as $filename => $fileContent) {
        $filePath = $outputDir . '/' . $filename;
        $fileDir = dirname($filePath);
        
        if (!is_dir($fileDir)) {
            mkdir($fileDir, 0755, true);
        }
        
        file_put_contents($filePath, $fileContent);
    }
    
    // Download images to output directory
    $imagesDir = $outputDir . '/images';
    mkdir($imagesDir, 0755, true);
    
    foreach ($images as $index => $imageUrl) {
        $imageData = @file_get_contents($imageUrl);
        if ($imageData) {
            file_put_contents($imagesDir . '/image-' . ($index + 1) . '.jpg', $imageData);
        }
    }
    
    // Step 5: Create ZIP
    $zipBuilder = new ZipBuilder();
    $zipPath = GENERATED_PATH . '/' . $domain . '-website.zip';
    $zipBuilder->create($outputDir, $zipPath);
    
    // Clean up temp directory
    deleteDirectory($outputDir);
    
    // Return success
    $downloadUrl = SITE_URL . '/generated/' . basename($zipPath);
    
    echo json_encode([
        'success' => true,
        'download_url' => $downloadUrl,
        'filename' => basename($zipPath)
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
    
    // Log error
    error_log('Site Builder Error: ' . $e->getMessage());
}

// Helper functions
function sanitizeDomain($domain) {
    $domain = strtolower(trim($domain));
    $domain = preg_replace('/^https?:\/\//', '', $domain);
    $domain = preg_replace('/^www\./', '', $domain);
    $domain = preg_replace('/\/.*$/', '', $domain);
    return $domain;
}

function sanitizeString($str) {
    return trim(strip_tags($str));
}

function isValidDomain($domain) {
    return preg_match('/^[a-zA-Z0-9][a-zA-Z0-9-]{0,61}[a-zA-Z0-9]?\.[a-zA-Z]{2,}$/', $domain);
}

function deleteDirectory($dir) {
    if (!is_dir($dir)) return;
    
    $files = array_diff(scandir($dir), ['.', '..']);
    foreach ($files as $file) {
        $path = $dir . '/' . $file;
        is_dir($path) ? deleteDirectory($path) : unlink($path);
    }
    rmdir($dir);
}
