<?php
/**
 * ZIP Builder
 * 
 * Creates ZIP archive from generated website files
 */

class ZipBuilder {
    
    /**
     * Create ZIP from directory
     */
    public function create($sourceDir, $zipPath) {
        if (!extension_loaded('zip')) {
            throw new Exception('ZIP extension not available');
        }
        
        $zip = new ZipArchive();
        
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new Exception('Failed to create ZIP file');
        }
        
        $sourceDir = realpath($sourceDir);
        
        if (!$sourceDir || !is_dir($sourceDir)) {
            throw new Exception('Source directory not found');
        }
        
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($sourceDir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath = substr($filePath, strlen($sourceDir) + 1);
                $zip->addFile($filePath, $relativePath);
            }
        }
        
        $zip->close();
        
        if (!file_exists($zipPath)) {
            throw new Exception('ZIP file was not created');
        }
        
        return $zipPath;
    }
    
    /**
     * Create ZIP from array of files
     */
    public function createFromArray($files, $zipPath) {
        $zip = new ZipArchive();
        
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new Exception('Failed to create ZIP file');
        }
        
        foreach ($files as $filename => $content) {
            $zip->addFromString($filename, $content);
        }
        
        $zip->close();
        
        return $zipPath;
    }
}
