<?php
/**
 * Template Engine
 * Generates complete HTML/CSS websites from content
 */

class TemplateEngine {
    private $templatesPath;
    private $templates = ['dark', 'warm', 'minimal', 'zen', 'clean'];
    
    public function __construct($templatesPath) {
        $this->templatesPath = $templatesPath;
    }
    
    public function selectTemplate($businessType) {
        $recommendations = [
            'repair_shop' => ['dark', 'clean'],
            'retail_store' => ['warm', 'clean'],
            'rental' => ['minimal', 'zen'],
            'fashion' => ['minimal', 'warm'],
            'wellness' => ['zen', 'minimal'],
            'beauty' => ['zen', 'warm'],
            'fitness' => ['dark', 'clean'],
            'medical' => ['clean', 'minimal'],
            'restaurant' => ['warm', 'dark'],
            'other' => $this->templates
        ];
        $options = $recommendations[$businessType] ?? $this->templates;
        return $options[array_rand($options)];
    }
    
    public function build($content, $template, $images, $domain) {
        $styles = $this->getStyles($template);
        return [
            'index.html' => $this->buildHome($content, $styles, $images, $domain),
            'about.html' => $this->buildAbout($content, $styles, $domain),
            'contact.html' => $this->buildContact($content, $styles, $domain),
            'privacy-policy.html' => $this->buildPrivacy($content, $styles, $domain),
            'terms-of-service.html' => $this->buildTerms($content, $styles, $domain),
            'css/style.css' => $this->buildCSS($styles),
            'js/main.js' => $this->buildJS()
        ];
    }
    
    private function getStyles($template) {
        $themes = [
            'dark' => [
                'name' => 'dark',
                'bg_body' => '#0f172a', 'bg_surface' => '#1e293b', 'bg_card' => '#1e293b',
                'text_primary' => '#f8fafc', 'text_secondary' => '#94a3b8', 'text_muted' => '#64748b',
                'accent' => '#38bdf8', 'accent_hover' => '#0ea5e9', 'accent_rgb' => '56, 189, 248',
                'border' => '#334155', 'btn_text' => '#0f172a',
                'font_heading' => "'Inter', sans-serif", 'font_body' => "'Inter', sans-serif",
                'hero_gradient' => 'radial-gradient(circle at top right, rgba(56, 189, 248, 0.1), transparent 50%)',
                'fonts_url' => 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap'
            ],
            'warm' => [
                'name' => 'warm',
                'bg_body' => '#fffbf5', 'bg_surface' => '#fef7ed', 'bg_card' => '#ffffff',
                'text_primary' => '#1c1917', 'text_secondary' => '#57534e', 'text_muted' => '#78716c',
                'accent' => '#d97706', 'accent_hover' => '#b45309', 'accent_rgb' => '217, 119, 6',
                'border' => '#e7e5e4', 'btn_text' => '#ffffff',
                'font_heading' => "'Playfair Display', serif", 'font_body' => "'Source Sans 3', sans-serif",
                'hero_gradient' => 'linear-gradient(135deg, #fef7ed 0%, #fef3c7 100%)',
                'fonts_url' => 'https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;600;700&family=Source+Sans+3:wght@400;500;600&display=swap'
            ],
            'minimal' => [
                'name' => 'minimal',
                'bg_body' => '#ffffff', 'bg_surface' => '#fafafa', 'bg_card' => '#ffffff',
                'text_primary' => '#09090b', 'text_secondary' => '#3f3f46', 'text_muted' => '#71717a',
                'accent' => '#18181b', 'accent_hover' => '#27272a', 'accent_rgb' => '24, 24, 27',
                'border' => '#e4e4e7', 'btn_text' => '#ffffff',
                'font_heading' => "'DM Sans', sans-serif", 'font_body' => "'DM Sans', sans-serif",
                'hero_gradient' => 'none',
                'fonts_url' => 'https://fonts.googleapis.com/css2?family=DM+Sans:wght@400;500;600;700&display=swap'
            ],
            'zen' => [
                'name' => 'zen',
                'bg_body' => '#faf5ff', 'bg_surface' => '#f5f3ff', 'bg_card' => '#ffffff',
                'text_primary' => '#1e1b4b', 'text_secondary' => '#4c1d95', 'text_muted' => '#6b7280',
                'accent' => '#7c3aed', 'accent_hover' => '#6d28d9', 'accent_rgb' => '124, 58, 237',
                'border' => '#e9d5ff', 'btn_text' => '#ffffff',
                'font_heading' => "'Outfit', sans-serif", 'font_body' => "'Outfit', sans-serif",
                'hero_gradient' => 'linear-gradient(135deg, #faf5ff 0%, #fce7f3 100%)',
                'fonts_url' => 'https://fonts.googleapis.com/css2?family=Outfit:wght@300;400;500;600;700&display=swap'
            ],
            'clean' => [
                'name' => 'clean',
                'bg_body' => '#f8fafc', 'bg_surface' => '#f1f5f9', 'bg_card' => '#ffffff',
                'text_primary' => '#0f172a', 'text_secondary' => '#475569', 'text_muted' => '#64748b',
                'accent' => '#0ea5e9', 'accent_hover' => '#0284c7', 'accent_rgb' => '14, 165, 233',
                'border' => '#e2e8f0', 'btn_text' => '#ffffff',
                'font_heading' => "'Plus Jakarta Sans', sans-serif", 'font_body' => "'Plus Jakarta Sans', sans-serif",
                'hero_gradient' => 'linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%)',
                'fonts_url' => 'https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:wght@400;500;600;700&display=swap'
            ]
        ];
        return $themes[$template] ?? $themes['clean'];
    }
    
    private function head($title, $desc, $styles, $domain) {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        return <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{$e($title)}</title>
    <meta name="description" content="{$e($desc)}">
    <meta name="robots" content="index, follow">
    <link rel="canonical" href="https://{$e($domain)}/">
    <meta property="og:title" content="{$e($title)}">
    <meta property="og:description" content="{$e($desc)}">
    <meta property="og:url" content="https://{$e($domain)}/">
    <meta property="og:type" content="website">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="{$styles['fonts_url']}" rel="stylesheet">
    <link rel="stylesheet" href="css/style.css">
</head>
HTML;
    }
    
    private function nav($businessName, $current = 'home') {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        $ha = $current === 'home' ? ' class="active"' : '';
        $aa = $current === 'about' ? ' class="active"' : '';
        $ca = $current === 'contact' ? ' class="active"' : '';
        return <<<HTML
<header class="header">
    <div class="container">
        <a href="index.html" class="logo">{$e($businessName)}</a>
        <button class="mobile-toggle" onclick="toggleMenu()" aria-label="Menu">☰</button>
        <nav class="nav" id="nav">
            <ul>
                <li><a href="index.html"{$ha}>Home</a></li>
                <li><a href="about.html"{$aa}>About</a></li>
                <li><a href="contact.html"{$ca}>Contact</a></li>
            </ul>
        </nav>
    </div>
</header>
HTML;
    }
    
    private function footer($content, $domain) {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        $bn = $e($content['business_name']);
        $tag = $e($content['footer']['tagline'] ?? '');
        $copy = $e($content['footer']['copyright'] ?? "© " . date('Y') . " {$bn}");
        $addr = $e($content['contact']['address'] ?? '');
        $phone = $e($content['contact']['phone'] ?? '');
        $email = $e($content['contact']['email'] ?? "contact@{$domain}");
        
        return <<<HTML
<footer class="footer">
    <div class="container">
        <div class="footer-grid">
            <div class="footer-col">
                <h4>{$bn}</h4>
                <p>{$tag}</p>
            </div>
            <div class="footer-col">
                <h4>Contact</h4>
                <p>{$addr}</p>
                <p><a href="tel:{$phone}">{$phone}</a></p>
                <p><a href="mailto:{$email}">{$email}</a></p>
            </div>
            <div class="footer-col">
                <h4>Legal</h4>
                <ul>
                    <li><a href="privacy-policy.html">Privacy Policy</a></li>
                    <li><a href="terms-of-service.html">Terms of Service</a></li>
                </ul>
            </div>
        </div>
        <div class="footer-bottom">
            <p>{$copy}</p>
        </div>
    </div>
</footer>
<div class="cookie-banner" id="cookieBanner">
    <div class="container">
        <p>We use cookies to improve your experience. See our <a href="privacy-policy.html">Privacy Policy</a>.</p>
        <button onclick="acceptCookies()" class="btn btn-sm">Accept</button>
    </div>
</div>
<script src="js/main.js"></script>
</body>
</html>
HTML;
    }
    
    private function buildHome($content, $styles, $images, $domain) {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        $bn = $content['business_name'];
        $title = $content['meta']['title'] ?? $bn;
        $desc = $content['meta']['description'] ?? '';
        
        $head = $this->head($title, $desc, $styles, $domain);
        $nav = $this->nav($bn, 'home');
        $footer = $this->footer($content, $domain);
        
        // Hero
        $heroH = $e($content['hero']['headline'] ?? 'Welcome');
        $heroS = $e($content['hero']['subheadline'] ?? '');
        
        // About
        $aboutT = $e($content['about']['title'] ?? 'About Us');
        $aboutC = nl2br($e($content['about']['content'] ?? ''));
        
        // Services
        $services = '';
        foreach (($content['services'] ?? []) as $s) {
            $st = $e($s['title'] ?? '');
            $sd = $e($s['description'] ?? '');
            $services .= "<div class=\"service-card\"><h3>{$st}</h3><p>{$sd}</p></div>";
        }
        
        // Process
        $processT = $e($content['process']['title'] ?? 'Our Process');
        $processI = $e($content['process']['intro'] ?? '');
        $steps = '';
        $n = 1;
        foreach (($content['process']['steps'] ?? []) as $step) {
            $pt = $e($step['title'] ?? '');
            $pd = $e($step['description'] ?? '');
            $steps .= "<div class=\"process-step\"><div class=\"step-num\">{$n}</div><div><h3>{$pt}</h3><p>{$pd}</p></div></div>";
            $n++;
        }
        
        // FAQ
        $faq = '';
        foreach (($content['faq'] ?? []) as $f) {
            $q = $e($f['question'] ?? '');
            $a = $e($f['answer'] ?? '');
            $faq .= "<div class=\"faq-item\"><button class=\"faq-q\" onclick=\"toggleFaq(this)\"><span>{$q}</span><span class=\"faq-icon\">+</span></button><div class=\"faq-a\"><p>{$a}</p></div></div>";
        }
        
        // Testimonials
        $test = '';
        foreach (($content['testimonials'] ?? []) as $t) {
            $tq = $e($t['quote'] ?? '');
            $ta = $e($t['author'] ?? '');
            $tl = $e($t['location'] ?? '');
            $test .= "<div class=\"testimonial\"><blockquote>\"{$tq}\"</blockquote><div class=\"t-author\"><strong>{$ta}</strong><span>{$tl}</span></div></div>";
        }
        
        return <<<HTML
{$head}
<body>
{$nav}
<main>
<section class="hero">
    <div class="container">
        <h1>{$heroH}</h1>
        <p class="hero-sub">{$heroS}</p>
        <div class="hero-btns">
            <a href="contact.html" class="btn btn-primary">Get in Touch</a>
            <a href="about.html" class="btn btn-outline">Learn More</a>
        </div>
    </div>
</section>

<section class="section">
    <div class="container">
        <div class="section-head"><h2>{$aboutT}</h2></div>
        <div class="about-text">{$aboutC}</div>
    </div>
</section>

<section class="section bg-alt">
    <div class="container">
        <div class="section-head"><h2>What We Offer</h2></div>
        <div class="services-grid">{$services}</div>
    </div>
</section>

<section class="section">
    <div class="container">
        <div class="section-head"><h2>{$processT}</h2><p>{$processI}</p></div>
        <div class="process-list">{$steps}</div>
    </div>
</section>

<section class="section bg-alt">
    <div class="container">
        <div class="section-head"><h2>What Clients Say</h2></div>
        <div class="testimonials-grid">{$test}</div>
    </div>
</section>

<section class="section">
    <div class="container">
        <div class="section-head"><h2>Frequently Asked Questions</h2></div>
        <div class="faq-list">{$faq}</div>
    </div>
</section>

<section class="section cta-section">
    <div class="container">
        <h2>Ready to Get Started?</h2>
        <p>Reach out today and let us know how we can help.</p>
        <a href="contact.html" class="btn btn-primary btn-lg">Contact Us</a>
    </div>
</section>
</main>
{$footer}
HTML;
    }
    
    private function buildAbout($content, $styles, $domain) {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        $bn = $content['business_name'];
        $head = $this->head("About | {$bn}", "Learn more about {$bn}", $styles, $domain);
        $nav = $this->nav($bn, 'about');
        $footer = $this->footer($content, $domain);
        $aboutT = $e($content['about']['title'] ?? 'About Us');
        $aboutC = nl2br($e($content['about']['content'] ?? ''));
        
        return <<<HTML
{$head}
<body>
{$nav}
<main>
<section class="page-hero"><div class="container"><h1>{$aboutT}</h1></div></section>
<section class="section"><div class="container"><div class="content-text">{$aboutC}</div></div></section>
<section class="section cta-section"><div class="container"><h2>Questions?</h2><p>We're here to help.</p><a href="contact.html" class="btn btn-primary">Contact Us</a></div></section>
</main>
{$footer}
HTML;
    }
    
    private function buildContact($content, $styles, $domain) {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        $bn = $content['business_name'];
        $head = $this->head("Contact | {$bn}", "Get in touch with {$bn}", $styles, $domain);
        $nav = $this->nav($bn, 'contact');
        $footer = $this->footer($content, $domain);
        $addr = $e($content['contact']['address'] ?? '');
        $phone = $e($content['contact']['phone'] ?? '');
        $email = $e($content['contact']['email'] ?? "contact@{$domain}");
        $hours = $e($content['contact']['hours'] ?? '');
        $intro = $e($content['contact']['intro'] ?? '');
        
        return <<<HTML
{$head}
<body>
{$nav}
<main>
<section class="page-hero"><div class="container"><h1>Contact Us</h1><p>{$intro}</p></div></section>
<section class="section">
    <div class="container">
        <div class="contact-grid">
            <div class="contact-info">
                <h2>Get in Touch</h2>
                <div class="contact-item"><h3>Address</h3><p>{$addr}</p></div>
                <div class="contact-item"><h3>Phone</h3><p><a href="tel:{$phone}">{$phone}</a></p></div>
                <div class="contact-item"><h3>Email</h3><p><a href="mailto:{$email}">{$email}</a></p></div>
                <div class="contact-item"><h3>Hours</h3><p>{$hours}</p></div>
            </div>
            <div class="contact-form-wrap">
                <h2>Send a Message</h2>
                <form class="contact-form">
                    <div class="form-group"><label>Name</label><input type="text" required></div>
                    <div class="form-group"><label>Email</label><input type="email" required></div>
                    <div class="form-group"><label>Phone</label><input type="tel"></div>
                    <div class="form-group"><label>Message</label><textarea rows="5" required></textarea></div>
                    <button type="submit" class="btn btn-primary">Send</button>
                </form>
            </div>
        </div>
    </div>
</section>
</main>
{$footer}
HTML;
    }
    
    private function buildPrivacy($content, $styles, $domain) {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        $bn = $content['business_name'];
        $head = $this->head("Privacy Policy | {$bn}", "Privacy Policy for {$bn}", $styles, $domain);
        $nav = $this->nav($bn);
        $footer = $this->footer($content, $domain);
        $pc = nl2br($e($content['privacy_policy'] ?? 'Privacy policy not available.'));
        $date = $content['generated_at'] ?? date('Y-m-d');
        
        return <<<HTML
{$head}
<body>
{$nav}
<main>
<section class="page-hero"><div class="container"><h1>Privacy Policy</h1><p>Last updated: {$date}</p></div></section>
<section class="section"><div class="container"><div class="legal-text">{$pc}</div></div></section>
</main>
{$footer}
HTML;
    }
    
    private function buildTerms($content, $styles, $domain) {
        $e = fn($s) => htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
        $bn = $content['business_name'];
        $head = $this->head("Terms of Service | {$bn}", "Terms of Service for {$bn}", $styles, $domain);
        $nav = $this->nav($bn);
        $footer = $this->footer($content, $domain);
        $tc = nl2br($e($content['terms_of_service'] ?? 'Terms not available.'));
        $date = $content['generated_at'] ?? date('Y-m-d');
        
        return <<<HTML
{$head}
<body>
{$nav}
<main>
<section class="page-hero"><div class="container"><h1>Terms of Service</h1><p>Last updated: {$date}</p></div></section>
<section class="section"><div class="container"><div class="legal-text">{$tc}</div></div></section>
</main>
{$footer}
HTML;
    }
    
    private function buildJS() {
        return <<<'JS'
function toggleMenu(){document.getElementById('nav').classList.toggle('active')}
function toggleFaq(b){var i=b.parentElement,a=i.classList.contains('active');document.querySelectorAll('.faq-item').forEach(function(f){f.classList.remove('active')});if(!a)i.classList.add('active')}
function acceptCookies(){localStorage.setItem('cookies','1');document.getElementById('cookieBanner').style.display='none'}
document.addEventListener('DOMContentLoaded',function(){if(localStorage.getItem('cookies'))document.getElementById('cookieBanner').style.display='none'});
JS;
    }
    
    private function buildCSS($s) {
        return <<<CSS
:root{--bg:{$s['bg_body']};--bg-alt:{$s['bg_surface']};--card:{$s['bg_card']};--text:{$s['text_primary']};--text2:{$s['text_secondary']};--muted:{$s['text_muted']};--accent:{$s['accent']};--accent2:{$s['accent_hover']};--border:{$s['border']};--btn-text:{$s['btn_text']};--font-h:{$s['font_heading']};--font-b:{$s['font_body']}}
*{margin:0;padding:0;box-sizing:border-box}
body{font-family:var(--font-b);background:var(--bg);color:var(--text);line-height:1.7}
a{color:var(--accent);text-decoration:none}a:hover{color:var(--accent2)}
h1,h2,h3,h4{font-family:var(--font-h);line-height:1.3;margin-bottom:1rem}
p{margin-bottom:1rem;color:var(--text2)}
.container{max-width:1200px;margin:0 auto;padding:0 24px}

.btn{display:inline-flex;align-items:center;padding:14px 28px;border-radius:8px;font-weight:600;cursor:pointer;border:2px solid transparent;transition:all .2s}
.btn-primary{background:var(--accent);color:var(--btn-text)}.btn-primary:hover{background:var(--accent2);color:var(--btn-text);transform:translateY(-2px)}
.btn-outline{border-color:var(--border);color:var(--text);background:transparent}.btn-outline:hover{border-color:var(--accent);color:var(--accent)}
.btn-lg{padding:18px 36px;font-size:1.1rem}.btn-sm{padding:10px 20px;font-size:.9rem}

.header{position:fixed;top:0;left:0;width:100%;background:var(--bg);border-bottom:1px solid var(--border);z-index:1000;padding:16px 0}
.header .container{display:flex;justify-content:space-between;align-items:center}
.logo{font-family:var(--font-h);font-size:1.5rem;font-weight:700;color:var(--text)}
.nav ul{display:flex;list-style:none;gap:32px}.nav a{color:var(--text2);font-weight:500}.nav a:hover,.nav a.active{color:var(--accent)}
.mobile-toggle{display:none;background:none;border:none;font-size:1.5rem;color:var(--text);cursor:pointer}

.hero{padding:160px 0 100px;background:{$s['hero_gradient']};text-align:center}
.hero h1{font-size:clamp(2.5rem,5vw,4rem);margin-bottom:1.5rem}
.hero-sub{font-size:1.25rem;color:var(--text2);max-width:600px;margin:0 auto 2rem}
.hero-btns{display:flex;gap:16px;justify-content:center;flex-wrap:wrap}

.page-hero{padding:140px 0 60px;background:var(--bg-alt);text-align:center}
.page-hero h1{font-size:clamp(2rem,4vw,3rem);margin-bottom:.5rem}.page-hero p{color:var(--muted)}

.section{padding:80px 0}.bg-alt{background:var(--bg-alt)}
.section-head{text-align:center;margin-bottom:48px}.section-head h2{font-size:clamp(1.75rem,3vw,2.5rem)}.section-head p{color:var(--muted);max-width:600px;margin:0 auto}

.about-text,.content-text,.legal-text{max-width:800px;margin:0 auto;font-size:1.1rem;line-height:1.8}

.services-grid{display:grid;grid-template-columns:repeat(auto-fit,minmax(280px,1fr));gap:24px}
.service-card{background:var(--card);padding:32px;border-radius:16px;border:1px solid var(--border);transition:transform .2s}
.service-card:hover{transform:translateY(-4px)}.service-card h3{font-size:1.25rem;margin-bottom:.75rem}.service-card p{margin-bottom:0}

.process-list{max-width:800px;margin:0 auto;display:grid;gap:24px}
.process-step{display:flex;gap:24px;align-items:flex-start}
.step-num{width:48px;height:48px;background:var(--accent);color:var(--btn-text);border-radius:50%;display:flex;align-items:center;justify-content:center;font-weight:700;font-size:1.25rem;flex-shrink:0}
.process-step h3{margin-bottom:.5rem}.process-step p{margin-bottom:0}

.testimonials-grid{display:grid;grid-template-columns:repeat(auto-fit,minmax(300px,1fr));gap:24px}
.testimonial{background:var(--card);padding:32px;border-radius:16px;border:1px solid var(--border)}
.testimonial blockquote{font-size:1.1rem;font-style:italic;color:var(--text2);margin-bottom:1.5rem;line-height:1.7}
.t-author strong{display:block;color:var(--text)}.t-author span{font-size:.9rem;color:var(--muted)}

.faq-list{max-width:800px;margin:0 auto}.faq-item{border-bottom:1px solid var(--border)}
.faq-q{width:100%;padding:20px 0;background:none;border:none;display:flex;justify-content:space-between;align-items:center;cursor:pointer;font-family:var(--font-b);font-size:1.1rem;font-weight:600;color:var(--text);text-align:left}
.faq-icon{font-size:1.5rem;color:var(--accent);transition:transform .2s}.faq-item.active .faq-icon{transform:rotate(45deg)}
.faq-a{max-height:0;overflow:hidden;transition:max-height .3s}.faq-item.active .faq-a{max-height:500px;padding-bottom:20px}.faq-a p{margin-bottom:0}

.cta-section{background:var(--accent);color:var(--btn-text);text-align:center}
.cta-section h2,.cta-section p{color:var(--btn-text)}.cta-section p{opacity:.9;margin-bottom:2rem}
.cta-section .btn-primary{background:var(--bg);color:var(--text)}.cta-section .btn-primary:hover{background:var(--bg-alt)}

.contact-grid{display:grid;grid-template-columns:1fr 1fr;gap:48px}
.contact-item{margin-bottom:1.5rem}.contact-item h3{font-size:1rem;color:var(--muted);margin-bottom:.25rem}.contact-item p{margin-bottom:0;color:var(--text)}
.contact-form-wrap{background:var(--bg-alt);padding:32px;border-radius:16px}
.form-group{margin-bottom:1.5rem}.form-group label{display:block;margin-bottom:.5rem;font-weight:500;color:var(--text2)}
.form-group input,.form-group textarea{width:100%;padding:14px 16px;border:1px solid var(--border);border-radius:8px;background:var(--bg);color:var(--text);font-family:var(--font-b);font-size:1rem}
.form-group input:focus,.form-group textarea:focus{outline:none;border-color:var(--accent)}

.footer{background:var(--bg-alt);padding:60px 0 20px;border-top:1px solid var(--border)}
.footer-grid{display:grid;grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:40px;margin-bottom:40px}
.footer-col h4{font-size:1rem;margin-bottom:1rem}.footer-col p,.footer-col li{color:var(--muted);font-size:.95rem;margin-bottom:.5rem}
.footer-col ul{list-style:none}.footer-col a{color:var(--muted)}.footer-col a:hover{color:var(--accent)}
.footer-bottom{padding-top:20px;border-top:1px solid var(--border);text-align:center}.footer-bottom p{color:var(--muted);font-size:.9rem;margin-bottom:0}

.cookie-banner{position:fixed;bottom:0;left:0;width:100%;background:var(--card);border-top:1px solid var(--border);padding:16px 0;z-index:1000}
.cookie-banner .container{display:flex;justify-content:space-between;align-items:center;gap:16px;flex-wrap:wrap}
.cookie-banner p{margin-bottom:0;font-size:.9rem}

@media(max-width:768px){
.mobile-toggle{display:block}
.nav{display:none;position:absolute;top:100%;left:0;width:100%;background:var(--bg);border-bottom:1px solid var(--border);padding:20px}
.nav.active{display:block}.nav ul{flex-direction:column;gap:16px}
.hero{padding:120px 0 60px}.section{padding:60px 0}
.contact-grid{grid-template-columns:1fr}
.process-step{flex-direction:column;text-align:center}.step-num{margin:0 auto}
}
CSS;
    }
}
