<?php
/**
 * Image Fetcher
 * 
 * Fetches relevant free images from Unsplash API
 */

class ImageFetcher {
    private $accessKey;
    private $apiUrl = 'https://api.unsplash.com/search/photos';
    
    public function __construct($accessKey) {
        $this->accessKey = $accessKey;
    }
    
    /**
     * Fetch images based on industry and business type
     */
    public function fetchImages($industry, $businessType, $count = 5) {
        // Build search queries
        $queries = $this->buildSearchQueries($industry, $businessType);
        $images = [];
        
        foreach ($queries as $query) {
            if (count($images) >= $count) break;
            
            $results = $this->search($query, $count - count($images));
            $images = array_merge($images, $results);
        }
        
        // If we don't have enough, try generic business images
        if (count($images) < $count) {
            $fallbackQueries = ['professional business', 'modern office', 'workspace'];
            foreach ($fallbackQueries as $query) {
                if (count($images) >= $count) break;
                $results = $this->search($query, $count - count($images));
                $images = array_merge($images, $results);
            }
        }
        
        // Ensure we have the requested count
        return array_slice($images, 0, $count);
    }
    
    /**
     * Build search queries based on business type
     */
    private function buildSearchQueries($industry, $businessType) {
        $queries = [];
        
        // Primary query from industry
        $queries[] = $industry;
        
        // Business type specific queries
        $typeQueries = [
            'repair_shop' => ['workshop tools', 'repair service', 'technician working'],
            'retail_store' => ['retail store interior', 'shop display', 'shopping'],
            'rental' => ['vacation rental', 'accommodation', 'cozy interior'],
            'fashion' => ['fashion clothing', 'boutique', 'style apparel'],
            'wellness' => ['spa wellness', 'relaxation', 'health lifestyle'],
            'beauty' => ['beauty salon', 'cosmetics', 'skincare'],
            'fitness' => ['gym fitness', 'exercise workout', 'athletic'],
            'medical' => ['healthcare clinic', 'medical office', 'doctor patient'],
            'restaurant' => ['restaurant food', 'dining', 'cuisine'],
            'travel' => ['travel destination', 'vacation', 'adventure'],
            'event' => ['event planning', 'celebration', 'party venue'],
            'photography' => ['photography camera', 'creative studio', 'photo shoot'],
            'education' => ['education learning', 'classroom', 'teaching'],
            'tech' => ['technology office', 'software development', 'digital'],
            'realestate' => ['real estate home', 'property house', 'interior design'],
            'pet' => ['pet care', 'animals', 'veterinary'],
            'professional' => ['professional office', 'business meeting', 'consulting'],
            'legal' => ['law office', 'legal documents', 'courthouse'],
            'finance' => ['financial planning', 'accounting', 'investment'],
            'construction' => ['construction site', 'building', 'contractor'],
            'automotive' => ['auto repair', 'car service', 'mechanic'],
            'cleaning' => ['cleaning service', 'housekeeping', 'professional cleaning'],
            'ecommerce' => ['online shopping', 'ecommerce', 'delivery package'],
            'other' => ['business professional', 'modern office', 'workspace']
        ];
        
        if (isset($typeQueries[$businessType])) {
            $queries = array_merge($queries, $typeQueries[$businessType]);
        }
        
        return $queries;
    }
    
    /**
     * Search Unsplash API
     */
    private function search($query, $perPage = 5) {
        if (empty($this->accessKey) || $this->accessKey === 'your-unsplash-access-key-here') {
            // Return placeholder images if no API key
            return $this->getPlaceholderImages($perPage);
        }
        
        $url = $this->apiUrl . '?' . http_build_query([
            'query' => $query,
            'per_page' => $perPage,
            'orientation' => 'landscape',
            'content_filter' => 'high'
        ]);
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Client-ID ' . $this->accessKey,
                'Accept-Version: v1'
            ],
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            return $this->getPlaceholderImages($perPage);
        }
        
        $data = json_decode($response, true);
        
        if (!isset($data['results']) || empty($data['results'])) {
            return $this->getPlaceholderImages($perPage);
        }
        
        $images = [];
        foreach ($data['results'] as $result) {
            // Use regular size (1080px width)
            $images[] = $result['urls']['regular'] ?? $result['urls']['small'];
        }
        
        return $images;
    }
    
    /**
     * Get placeholder images when API is not available
     */
    private function getPlaceholderImages($count) {
        $placeholders = [];
        $colors = ['0f172a', '1e293b', '334155', '1e3a5f', '1a365d'];
        
        for ($i = 0; $i < $count; $i++) {
            $color = $colors[$i % count($colors)];
            $placeholders[] = "https://via.placeholder.com/1200x800/{$color}/94a3b8?text=Image+" . ($i + 1);
        }
        
        return $placeholders;
    }
    
    /**
     * Download image to local file
     */
    public function downloadImage($url, $savePath) {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 60
        ]);
        
        $imageData = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200 && $imageData) {
            file_put_contents($savePath, $imageData);
            return true;
        }
        
        return false;
    }
}
